from django.contrib import admin
from .models import Category, Product

@admin.register(Product)
class ProductAdmin(admin.ModelAdmin):
    list_display = ['name', 'category', 'volume', 'in_stock', 'is_active', 'has_image']
    list_filter = ['category', 'in_stock', 'is_active', 'created']
    search_fields = ['name', 'description', 'ingredients']
    prepopulated_fields = {'slug': ('name',)}
    readonly_fields = ['created', 'updated', 'image_preview']
    
    fieldsets = (
        (None, {
            'fields': ('name', 'slug', 'category', 'description')
        }),
        ('Images', {
            'fields': ('image', 'image_url', 'image_preview'),
            'classes': ('collapse',)
        }),
        ('Product Details', {
            'fields': ('form', 'volume', 'ingredients', 'usage_instructions')
        }),
        ('Status', {
            'fields': ('in_stock', 'is_active')
        }),
        ('Timestamps', {
            'fields': ('created', 'updated'),
            'classes': ('collapse',)
        })
    )
    
    def image_preview(self, obj):
        if obj.has_image:
            return f'<img src="{obj.display_image}" style="max-height: 200px; max-width: 200px;" />'
        return "No image available"
    image_preview.allow_tags = True
    image_preview.short_description = 'Image Preview'

@admin.register(Category)
class CategoryAdmin(admin.ModelAdmin):
    list_display = ['name', 'slug']
    prepopulated_fields = {'slug': ('name',)}